// -------------------------------------------------------------------------
// VNX_Atest.cpp : This is an example console application which uses the
// Lab Brick attenuator API to control one or more attenuators
//
//	This example shows how to create multiple attenuation ramps that start together on multiple attenuators
//	Note that some settings require an LDA with the bidirectional ramp feature.
//
// -------------------------------------------------------------------------
#include "stdafx.h"


#include "VNX_atten.h"

#include <string>
#include <algorithm>
using namespace std;


// ------------------------------- Equates -----------------------------------------------
#define SWP_DIRECTION		0x04	// MASK: bit = 0 for ramp up, 1 for ramp down 
#define SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous ramping
#define SWP_ONCE			0x01	// MASK: bit = 1 for single ramp
#define SWP_BIDIRECTIONALLY	0x10	// MASK: bit = 1 for bidirectional ramps (V2 LDA's only)
 

// ------------------------------- Allocations -------------------------------------------

static DEVID MyDevices[MAXDEVICES];			// I have statically allocated this array for convenience
											// It holds a list of device IDs for the connected devices
											// They are stored starting at MyDevices[0]

static char MyDeviceName[MAX_MODELNAME];	// NB -- its a char array for now, no support for wchars, etc.

static string sDevName			= "ATN-001";// device name string
static bool gbWantOneDevice		= FALSE;

static int DevNum				= 0;		// the device we should work with.
static int DevRange				= 1;		// the number of devices we'll send the command to
static int NumDevices			= 0;		// used to store the actual number of devices found


// --------------------------- Variables -------------------------------------------------

static long IdleTime			= 1;		// default idle time is 1 ms
static long HoldTime			= 1;		// default hold time is 1 ms
static long AStart				= 0;		// default atten start level is 0 db.
static long AStop				= 252;		// default atten stop, for most devices this is 63 db so we use that.
static long Dwell				= 1000;		// default dwell time is 1 second for first ramp phase
static long Dwell2				= 1000;		// default dwell time is 1 second for second ramp phase (V2 LDA's only)
static long AStep				= 2;		// default step size is .5db, some LDA's have larger minimum steps
static long AStep2				= 2;		// default second phase step size for LDA's that support bidirectional ramps

static long Attenuation			= 0;		// default attenuation is 0db, encoding is in .25 db steps

static long Power				= 0;		// sets absolute power level, clipped by device's max calibrated power
											// it is in units of .25 db

static long SerialNumber		= 0;		// used to hold the serial number for the get serial number command

static int RFOnOff				= 1;		// really used as a bool -- if non zero, turn on the RF output

static int Sweep_mode			= 0;		// used to control the sweep mode
static int GetParam				= 0;		// the low byte is the GET command byte

BOOL gbWantSetIdle				= FALSE;
BOOL gbWantSetHold				= FALSE;
BOOL gbWantSetAStart			= FALSE;
BOOL gbWantSetAStop				= FALSE;
BOOL gbWantSetDwell				= FALSE;
BOOL gbWantSetDwell2			= FALSE;
BOOL gbWantStartSweep			= FALSE;
BOOL gbWantSetAStep				= FALSE;
BOOL gbWantSetAStep2			= FALSE;
BOOL gbWantSetAttenuation		= FALSE;
BOOL gbWantSaveSettings			= FALSE;
BOOL gbWantGetParam				= FALSE;
BOOL gbBatchMode				= FALSE;
BOOL gbWantMaxPower				= FALSE;
BOOL gbWantSetRFOnOff			= FALSE;
BOOL gbQuietMode				= FALSE;

// -------------------------------- Command Line routines --------------------------------
void PrintHelp()
{
	printf("Vaunix Multiple Attenuator Demonstration\n");
	printf("\n");
	printf("Hit CTRL+C to exit\n");
	printf("\n");

	printf(" --- Overall modes and device selection. Defaults to first device ---\n");
	printf("  -d i n 	Select the devices to work with, i is the device number (1,2,3, etc.)\n");
	printf("     		and n is the number of devices to apply the command to.\n");
	printf("     		-d 1 2 applies the commands to attenuators 1 and 2.\n");
	printf("     		-d 2 3 applies the commands to attenuators 2, 3 and 4.\n");
	printf("  -y		Save the current settings in the device.\n");
	printf("\n");
	printf("  -b		Batch mode, exit immediately after sending commands to the Lab Bricks.\n");
	printf("  -q		Quiet mode, skip most outputs.\n");
	printf("\n");

	printf(" --- Commands to set parameters and start ramp --- \n");
	printf("  -a nn     Set attenuator, nn is attenuation in .25db units\n");
	printf("            .25 db = 1, 1 db = 4, 10 db = 40 \n");
	printf("  -w nn     Set idle time between attenuator ramps, nn is time in ms.\n");
	printf("  -h nn     Set hold time between ramp phases\n");
	printf("  -s nn     Set ramp start value, nn is start value\n");
	printf("  -e nn     Set ramp end value, nn is end value, p is ramp phase\n");
	printf("  -t p nn   Set time to dwell on each attenuation value, nn is time in ms., p is ramp phase 1 or 2\n");

	printf("  -i p nn   Set attenuation ramp increment, nn is the increment\n");
	printf("            in .25 db units. p is ramp phase 1 or 2\n");
	printf("  -g n      Start a ramp, 1 = once upwards, 2 = continuous upwards\n");
	printf("            5 = once down, 6 = continuous down, 17 = bidirectional once,\n");
	printf("            18 = continuous bidirectional ramps, 0 to stop\n");

	printf("\n");


}
//.-----------------------------------------------------------------------------------
char MakeLower(char &c)
{
	return c = tolower(c);
}

// -----------------------------------------------------------------------------------
// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should abort for another reason.
BOOL ParseCommandLine(int argc, char *argv[])
{
	int RampPhase;

	enum { wantDash, wantDevSubstring, wantIdle, wantAStart, wantAStop, wantDwell, wantAStep,
		   wantAtten, wantSetRFOnOff, wantSweep, wantGetParam, wantDevID, wantDevRange,
		   wantDwell2, wantAStep2, wantHold, wantDwellPhase, wantStepPhase } state = wantDash;

	for (int i = 1; i < argc; ++i) {
		// Convert each argument to lowercase
		string thisParam(argv[i]);
		for_each(thisParam.begin(), thisParam.end(), MakeLower);

		BOOL bHasDash = ('-' == thisParam[0]);
		if (bHasDash) {
			if (state != wantDash) {
				PrintHelp();
				return FALSE;
			}

			// remove the dash from the front of the string
			thisParam = string(thisParam.begin() + 1, thisParam.end());

			// Identify the command line arguments
			if ("d" == thisParam) {
				state = wantDevID;
			}
			else if ("w" == thisParam) {
				gbWantSetIdle = TRUE;
				state = wantIdle;
			}
			else if ("s" == thisParam) {
				gbWantSetAStart = TRUE;
				state = wantAStart;
			}
			else if ("e" == thisParam) {
				gbWantSetAStop = TRUE;
				state = wantAStop;
			}
			else if ("t" == thisParam) {
				state = wantDwellPhase;
			}
			else if ("i" == thisParam) {
				state = wantStepPhase;
			}
			else if ("a" == thisParam) {
				gbWantSetAttenuation = TRUE;
				state = wantAtten;
			}
			else if ("g" == thisParam) {
				gbWantStartSweep = TRUE;
				state = wantSweep;
			}
			else if ("r" == thisParam) {
				gbWantSetRFOnOff = TRUE;
				state = wantSetRFOnOff;
			}
			else if ("y" == thisParam) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if ("b" == thisParam) {
				gbBatchMode = TRUE;
				state = wantDash;
			}
			else if ("q" == thisParam) {
				gbQuietMode = TRUE;
				state = wantDash;
			}
			else if ("h" == thisParam) {
				gbWantSetHold = TRUE;
				state = wantHold;
			}

			else {
				// this case is for "-?" and any argument we don't recognize
				PrintHelp();
				return FALSE;	// don't continue
			}
		} else {


			// save the whole substring and do conversions for each argument type

			switch (state){

			case wantDwellPhase:
				RampPhase = atol(thisParam.c_str());
				if (RampPhase == 1){
					gbWantSetDwell = TRUE;
					state = wantDwell;
				}
				else if (RampPhase == 2){
					gbWantSetDwell2 = TRUE;
					state = wantDwell2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantStepPhase:
				RampPhase = atol(thisParam.c_str());
				if (RampPhase == 1){
					gbWantSetAStep = TRUE;
					state = wantAStep;
				}
				else if (RampPhase == 2){
					gbWantSetAStep2 = TRUE;
					state = wantAStep2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantIdle:
				IdleTime = atol(thisParam.c_str());		// convert to a long
				state = wantDash;
				break;

			case wantHold:
				HoldTime = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantDevID:
				DevNum = atoi(thisParam.c_str());
				state = wantDevRange;
				break;

			case wantDevRange:
				DevRange = atoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStart:
				AStart = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStop:
				AStop = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantDwell:
				Dwell = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantDwell2:
				Dwell2 = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStep:
				AStep = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStep2:
				AStep2 = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantAtten:
				Attenuation = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantSetRFOnOff:
				RFOnOff = atol(thisParam.c_str());
				state = wantDash;
				break;

			case wantSweep:
				Sweep_mode = atoi(thisParam.c_str());
				state = wantDash;
				break;


			}


		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}

// -------------------------------- Program Main -----------------------------------------

int main(int argc, char* argv[])
{
	int i, j;
	int iDev;
	int itemp;
	bool bTemp;
	float ftemp;

	if (!gbQuietMode) printf("Lab Brick Multi Attenuator Demonstration Program\n");

	if (!ParseCommandLine(argc, argv))
		return 0;

//	 -- convert the user's device number to our internal MyDevices array index and check our device range --

	DevNum = DevNum - 1;

	if (DevNum < 0) DevNum = 0;
	if (DevRange < 1) DevRange = 1;
	if (DevRange > MAXDEVICES) DevRange = MAXDEVICES;

	if (DevNum  > MAXDEVICES - 1) DevNum = MAXDEVICES - 1;
	if ((DevNum + DevRange) > MAXDEVICES) DevRange = MAXDEVICES - DevNum;

	// at this point our device starting index and number of devices should be reasonable...
	if (!gbQuietMode) printf("Starting device number = %d, using %d devices\n", DevNum+1, DevRange);

	// --- if TestMode = TRUE then the dll will fake the hardware ---
	fnLDA_SetTestMode(FALSE);
	
	i = fnLDA_GetNumDevices();

	if (i == 0){
		printf("No device found\n");
	}

	if (i == 1){
		if (!gbQuietMode) printf("Found %d Device\n", i);

	}
	else {
		if (!gbQuietMode) printf("Found %d Devices\n", i);
	}

	// -- warn the user if he or she expects more devices than we have --
	if (DevRange > i){
		printf(" Warning - not enough attenuators are connected\n");
	}

	NumDevices = fnLDA_GetDevInfo(MyDevices);

	if (!gbQuietMode) printf("Got Device Info for %d Device[s]\n", NumDevices);

	
	if ( NumDevices > 0)	// do we have a device?
	{
		for (j = 0; j < NumDevices; j++){

		// --- print out the first device's name ---
		if (!gbQuietMode){
			itemp = fnLDA_GetModelName(MyDevices[j], MyDeviceName);
			printf("Device %d is an %s \n", j+1, MyDeviceName);
		}

		// --- print out the device's serial number ---
		if (!gbQuietMode){
			itemp = fnLDA_GetSerialNumber(MyDevices[j]);
			printf("Device %d has serial number %d \n", j+1, itemp);
		}


		// --- We need to init the device (open it) before we can do anything else ---
		itemp = fnLDA_InitDevice(MyDevices[j]);

		if (itemp){
			printf("InitDevice returned error code %x\n", itemp);
		}
		
		// --- Lets see if we got the device's parameters ---
		if (!gbQuietMode){
			itemp = fnLDA_GetRampStart(MyDevices[j]);
			ftemp = itemp * .25;
			printf("Ramp Start Level = %.2f db\n", ftemp);
		
			itemp = fnLDA_GetRampEnd(MyDevices[j]);
			ftemp = itemp * .25;
			printf("Ramp End Level = %.2f db\n", ftemp);
		
			itemp = fnLDA_GetAttenuationStep(MyDevices[j]);
			ftemp = itemp * .25;
			printf("First Phase Ramp Attenuation Step Size = %.2f db\n", ftemp);

			if (fnLDA_GetFeatures(MyDevices[j]) > 0){
				itemp = fnLDA_GetAttenuationStepTwo(MyDevices[j]);
				ftemp = itemp * .25;
				printf("Second Phase Ramp Attenuation Step Size = %.2f db\n", ftemp);
			}

			itemp = fnLDA_GetDwellTime(MyDevices[j]);
			printf("First Phase Ramp Dwell Time = %d\n", itemp);

			if (fnLDA_GetFeatures(MyDevices[j]) > 0){
				itemp = fnLDA_GetDwellTimeTwo(MyDevices[j]);
				printf("Second Phase Ramp Dwell Time = %d\n", itemp);
			}

			itemp = fnLDA_GetIdleTime(MyDevices[j]);
			printf("Ramp Idle Time = %d\n", itemp);

			if (fnLDA_GetFeatures(MyDevices[j]) > 0){
				itemp = fnLDA_GetHoldTime(MyDevices[j]);
				printf("Ramp Hold Time = %d\n", itemp);
			}

			itemp = fnLDA_GetMinAttenuation(MyDevices[j]);
			ftemp = itemp * .25;
			printf("Minimum Attenuation = %.2f db\n", ftemp);
		
			itemp = fnLDA_GetMaxAttenuation(MyDevices[j]);
			ftemp = itemp * .25;
			printf("Maximum Attenuation = %.2f db\n", ftemp);

			// --- Show if the RF output is on ---
			itemp = fnLDA_GetRF_On(MyDevices[j]);

			if (itemp != 0){
				printf("RF ON\n");
			}
			else{
				printf("RF OFF\n");
			}

		} // end of our quiet mode case

		} // end of the for loop over the devices

		// if the user is trying to control a device we don't have, then quit now

		if (DevNum > NumDevices - 1){
			for (j = 0; j < NumDevices; j++)
			{
				itemp = fnLDA_CloseDevice(MyDevices[j]);
			}
			printf("First selected device is not attached, exiting.\n");
			return 0;			// quit - nothing else to do
		}

		// if the user is trying to control more devices than we have, reduce the number of devices in the group
		if ((DevNum + DevRange) > NumDevices){
			DevRange = NumDevices - DevNum;
			printf("Not enough attenuators connected, using %d devices.\n", DevRange);
		}

		// ------------- Now we'll set the requested device or devices with new parameters -------------

		if (!gbQuietMode)printf("Setting the attenuator parameters..\n");

		for (iDev = DevNum; iDev < DevNum + DevRange; iDev++)
		{
			// --- Lets set the attenuation first ---
			if (gbWantSetAttenuation)
			{
				ftemp = (float) Attenuation / 4;
				if (!gbQuietMode) printf("Setting the attenuation for device %d to %.2f db\n", iDev+1, ftemp);
				itemp = fnLDA_SetAttenuation(MyDevices[iDev], Attenuation);
			}

			// --- and then do whatever else the user requested ---
		
			if (gbWantSetDwell)
			{
				if (!gbQuietMode) printf("Setting the first phase dwell time for device %d to %d \n", iDev+1, Dwell);
				itemp = fnLDA_SetDwellTime(MyDevices[iDev], Dwell);
			}

			if (gbWantSetAStart)
			{
				ftemp = (float) AStart / 4;
				if (!gbQuietMode) printf("Setting the ramp start for device %d to %.2f db \n", iDev+1, ftemp);
				itemp = fnLDA_SetRampStart(MyDevices[iDev], AStart);
			}
	
			if (gbWantSetAStop)
			{
				ftemp = (float) AStop / 4;
				if (!gbQuietMode) printf("Setting ramp end for device %d to %.2f db \n", iDev+1, ftemp);
				itemp = fnLDA_SetRampEnd(MyDevices[iDev], AStop);
			}

			if (gbWantSetAStep)
			{
				ftemp = (float) AStep / 4;
				if (!gbQuietMode) printf("Setting the first phase attenuation step for device %d to %.2f db \n", iDev+1, ftemp);
				itemp = fnLDA_SetAttenuationStep(MyDevices[iDev], AStep);
			}

			if (gbWantSetIdle)
			{
				if (!gbQuietMode) printf("Setting the idle time between ramps for device %d to %d ms. \n", iDev+1, IdleTime);
				itemp = fnLDA_SetIdleTime(MyDevices[iDev], IdleTime);
			}

			if (gbWantSetRFOnOff)
			{
				if (RFOnOff == 0)
				{
					bTemp = FALSE;
					if (!gbQuietMode) printf("Setting the maximum attenuation (RF OFF) for device %d\n", iDev+1);				
				}
				else
				{
					bTemp = TRUE;
					if (!gbQuietMode) printf("Setting the minimum attenuation (RF ON) for device %d\n", iDev+1);
				}

				itemp = fnLDA_SetRFOn(MyDevices[iDev], bTemp);
			}

			// if we have a V2 Lab Brick, send it the additional commands
			if (fnLDA_GetFeatures(MyDevices[iDev]) > 0)
			{
				if (gbWantSetAStep2)
				{
					ftemp = (float) AStep2 / 4;
					if (!gbQuietMode) printf("Setting the second phase attenuation step for device %d to %.2f db \n", iDev+1, ftemp);
					itemp = fnLDA_SetAttenuationStepTwo(MyDevices[iDev], AStep2);
				}
				if (gbWantSetDwell2)
				{
					if (!gbQuietMode) printf("Setting the second phase dwell time for device %d to %d ms \n", iDev+1, Dwell2);
					itemp = fnLDA_SetDwellTimeTwo(MyDevices[iDev], Dwell2);
				}
				if (gbWantSetHold)
				{
					if (!gbQuietMode) printf("Setting the hold time between ramp phases for device %d to %d ms \n", iDev+1, HoldTime);
					itemp = fnLDA_SetHoldTime(MyDevices[iDev], HoldTime);
				}
			}

			if (gbWantSaveSettings)
			{
				if (!gbQuietMode) printf("Saving the settings for device %d\n", iDev+1);
				fnLDA_SaveSettings(MyDevices[iDev]);
			}

		}	// this is the end of our for loop over devices for the general commands

		// -- For ramps we first set the parameters, then send the actual commands to start the ramps
		//	  grouping these commands reduces the latency between the ramps on each attenuator
		for (iDev = DevNum; iDev < DevNum + DevRange; iDev++)
		{
			if (gbWantStartSweep)
				{
				// --- first we'll figure out what the user wants us to do ---

				if (Sweep_mode == 0)
				{
					if (!gbQuietMode) printf("Stopping the Attenuation Ramp\n");
					itemp = fnLDA_StartRamp(MyDevices[iDev], FALSE);

				}
				else
				{

					// --- The user wants to start some kind of an attenuation ramp ---
					if (Sweep_mode & SWP_DIRECTION)
						{ bTemp = FALSE;}
					else
						{ bTemp = TRUE;}	// NB -- the flag is TRUE for "up" in the Set...Direction call.
											// but the old test program uses a 0 bit for up, and a 1 bit for down...

					itemp = fnLDA_SetRampDirection(MyDevices[iDev], bTemp);

					// --- and now we'll do the mode - one time or repeated ---
					if (Sweep_mode & SWP_ONCE)
						{ bTemp = FALSE;}
					else
						{ bTemp = TRUE;}	// NB -- the flag is TRUE for "repeated" in the SetSweepMode call.
											// but the old test program encodes the modes differently

					itemp = fnLDA_SetRampMode(MyDevices[iDev], bTemp);

					// --- and then the bidirectional ramp control if the device is a V2 device
					if (fnLDA_GetFeatures(MyDevices[iDev]) > 0)
					{
						if (Sweep_mode & SWP_BIDIRECTIONALLY)
							{ bTemp = TRUE;}				// the command line has true for bidirectional 
						else								// as does the actual HW command...
							{ bTemp = FALSE;} 

						printf("Bidirection mode set to %x \n", bTemp);
						itemp = fnLDA_SetRampBidirectional(MyDevices[iDev], bTemp);
					}

					if (!gbQuietMode) printf("Starting an attenuation ramp for device %d\n", iDev+1);
					itemp = fnLDA_StartRamp(MyDevices[iDev], TRUE);
				}
			}
		} // this is the end of our for loop over selected devices for the ramp command

		
		// -- Lets report on the device's operation for a little while, unless we are in batch mode

		if (!gbBatchMode)
		{
			j = 0;
			while (j < 40)
			{
				for (iDev = DevNum; iDev < DevNum + DevRange; iDev++)
				{
					ftemp = ((float) fnLDA_GetAttenuation(MyDevices[iDev])) / 4;
					printf("Attenuation = %.2f db for device %d\n", ftemp, iDev+1);
				}
				printf("\n");
				Sleep(500);		// wait for 1/2 second
				j++;
			}
		
		} // end of if not batch mode

		// -- we've done whatever the user wanted, time to close the devices
		
		for (j = 0; j < i; j++)
		{
			itemp = fnLDA_CloseDevice(MyDevices[j]);
		}

	} // end of if ( i > 0 ) -- "we have a device"

	return 0;
}

// --------------------------- end of main ----------------------------------
